/*:
 /*:
 * @target MZ
 * @plugindesc Tic Tac Toe game for RPG Maker MZ
 * @author ToshaAngel
 * @version v1.0.0
 *
 * @param XImage
 * @text X Image
 * @desc Specify the file name for the X image (place file in img/pictures/tictactoe folder)
 * @default tictactoe/X_Image
 * @dir img/pictures/
 * @type file
 *
 * @param XSound
 * @text X Sound
 * @desc Specify the file name for the X sound (place file in audio/se folder)
 * @default X_Sound
 * @dir audio/se/
 * @type file
 *
 * @param OImage
 * @text O Image
 * @desc Specify the file name for the O image (place file in img/pictures/tictactoe folder)
 * @default tictactoe/O_Image
 * @dir img/pictures/
 * @type file
 *
 * @param OSound
 * @text O Sound
 * @desc Specify the file name for the O sound (place file in audio/se folder)
 * @default O_Sound
 * @dir audio/se/
 * @type file
 *
 * @param HighlightImage
 * @text Highlight Image
 * @desc Specify the file name for the highlight image (place file in img/pictures/tictactoe folder)
 * @default tictactoe/Highlight_Image
 * @dir img/pictures/
 * @type file
 *
 * @param BackgroundImage
 * @text Background Image
 * @desc Specify the file name for the background image (place file in img/pictures/tictactoe folder)
 * @default tictactoe/Background_Image
 * @dir img/pictures/
 * @type file
 *
 * @param WinSwitch
 * @text Win Switch
 * @desc Switch that will be activated upon player's victory.
 * @type switch
 *
 * @param LoseSwitch
 * @text Lose Switch
 * @desc Switch that will be activated upon player's loss.
 * @type switch
 *
 * @param TieSwitch
 * @text Tie Switch
 * @desc Switch that will be activated upon a tie.
 * @type switch
 *
 * @param BoardOpacity
 * @text Board Opacity
 * @desc Set the opacity of the game board from 0 to 255.
 * @default 0
 * @type number
 * @min 0
 * @max 255
 *
 * @command StartTicTacToe
 * @text Start Tic Tac Toe
 * @desc Starts the Tic Tac Toe game.
 *
 * @arg BGM
 * @text Background Music
 * @desc Specify the background music for the game.
 * @default
 * @dir audio/bgm/
 * @type file
 *
 * @arg ShowEndGameMessage
 * @text Show End Game Message
 * @desc If set to true, a message will be shown at the end of the game.
 * @type boolean
 * @default true
 *
 * @arg WinMessage
 * @text Win Message
 * @desc Message shown upon player's victory.
 * @type string
 * @default You won!
 *
 * @arg LoseMessage
 * @text Lose Message
 * @desc Message shown upon player's loss.
 * @type string
 * @default You lost!
 *
 * @arg TieMessage
 * @text Tie Message
 * @desc Message shown upon a tie.
 * @type string
 * @default It's a tie!
 *
* @help
*
*
 * ___________          .__              
 * \__    ___/___  _____|  |__ _____     
 *   |    | /  _ \/  ___/  |  \\__  \    
 *   |    |(  <_> )___ \|   Y  \/ __ \_  
 *   |____| \____/____  >___|  (____  /  
 *                    \/     \/     \/   
 *    _____                        .__   
 *   /  _  \   ____    ____   ____ |  |  
 *  /  /_\  \ /    \  / ___\_/ __ \|  |  
 * /    |    \   |  \/ /_/  >  ___/|  |__
 * \____|__  /___|  /\___  / \___  >____/
 *         \/     \//_____/      \/   
 * 
 *
 * Provides an enhanced tic-tac-toe mini-game for RPG Maker MZ,
 * adding a new level of interactivity to your project. With this plugin, you can
 * integrate tic-tac-toe directly into your game, customize the visual
 * and audio aspects of the game, and use the game's outcomes to influence events in your world.
 * 
 *
 * - Fully customizable images for X and O, allowing integration into any art style.
 * - Customizable sound effects for player and AI moves, enhancing the audio experience.
 * - Ability to set background music to create atmosphere during gameplay.
 * - Control over the transparency of the game board.
 * - Customizable switches (WinSwitch, LoseSwitch, TieSwitch) that can be
 *   activated based on the game's outcome, allowing you to trigger various events in your project.
 * - Support for end-game messages for victory, defeat, or draw.
 * 
 * 
 * Examples:
 * 1. Tavern mini-game where the player can play tic-tac-toe with an NPC.
 * 2. Dungeon puzzle where the player must defeat AI in tic-tac-toe to open a secret door.
 * 3. In-game event where the outcome of tic-tac-toe affects the storyline or relationships with other characters.
 *

 */


(function () {
  const parameters = PluginManager.parameters("ToshA_TicTacToe");
  const xImage = parameters["XImage"];
  const oImage = parameters["OImage"];
  const backgroundImage = parameters["BackgroundImage"];
  const winSwitch = Number(parameters["WinSwitch"]);
  const loseSwitch = Number(parameters["LoseSwitch"]);
  const tieSwitch = Number(parameters["TieSwitch"]);
  const winMessage = parameters["WinMessage"];
  const loseMessage = parameters["LoseMessage"];
  const tieMessage = parameters["TieMessage"];
  const xSound = parameters["XSound"];
const oSound = parameters["OSound"];


PluginManager.registerCommand("ToshA_TicTacToe", "StartTicTacToe", args => {
    SceneManager.push(Scene_TicTacToe);
    Scene_TicTacToe.prototype.showEndGameMessage = args.ShowEndGameMessage === "true";
    Scene_TicTacToe.prototype.customWinMessage = args.WinMessage;
    Scene_TicTacToe.prototype.customLoseMessage = args.LoseMessage;
    Scene_TicTacToe.prototype.customTieMessage = args.TieMessage;
    Scene_TicTacToe.prototype.bgm = args.BGM;
});


  class Scene_TicTacToe extends Scene_Base {
start() {
    super.start();
    this.savedBgm = AudioManager.saveBgm();
    this.savedBgs = AudioManager.saveBgs();
    AudioManager.stopBgs();

    if (this.bgm) {
        AudioManager.playBgm({name: this.bgm, volume: 100, pitch: 100, pan: 0});
    }
}


terminate() {
    super.terminate();
    if (this.savedBgm) {
        AudioManager.replayBgm(this.savedBgm);
    }
    if (this.savedBgs) {
        AudioManager.replayBgs(this.savedBgs);
    }
}


    constructor() {
      super();
      this._cursorX = 0;
      this._cursorY = 0;
      this.gameOver = false;
    }
    createBoardWindows() {
      this.boardWindows = [];
      const cellWidth = 162; // Ширина клетки
      const cellHeight = 114; // Высота клетки
      this.cellPositions = [  // Позиции клеток
        [
          { x: 120, y: 55 },  // Левая верхняя
          { x: 360, y: 65 },  // Средняя верхняя
          { x: 595, y: 62 },  // Правая верхняя
        ],
        [
          { x: 80, y: 225 },  // Левая средняя ... ...
          { x: 310, y: 230 },
          { x: 545, y: 235 },
        ],
        [
          { x: 30, y: 385 },
          { x: 270, y: 390 },
          { x: 510, y: 400 },
        ],
      ];

      for (let i = 0; i < 3; i++) {
        for (let j = 0; j < 3; j++) {
          const pos = cellPositions[i][j];
          const window = new Window_Base(
            new Rectangle(pos.x, pos.y, cellWidth, cellHeight),
          );
          window.opacity = Number(parameters["BoardOpacity"]);
          this.addWindow(window);
          this.boardWindows.push(window);
        }
      }
    }
makeMove(x, y) {
    if (!this.board[y][x]) {
        this.board[y][x] = this.currentPlayer;
    const soundName = this.currentPlayer === "X" ? xSound : oSound;
    if (soundName) {
        AudioManager.playSe({name: soundName, pan: 0, pitch: 100, volume: 90});
    }

    if (this.checkWin()) {
      this.endGame(`${this.currentPlayer} wins!`);
    } else if (this.checkTie()) {
      this.endGame(`It's a tie!`);
    } else {
      this.currentPlayer = this.currentPlayer === "X" ? "O" : "X";
    }
  }
}

    processMouseInput() {
      if (TouchInput.isTriggered()) {
        const x = TouchInput.x;
        const y = TouchInput.y;

        for (let i = 0; i < this.cellPositions.length; i++) {
          for (let j = 0; j < this.cellPositions[i].length; j++) {
            const cell = this.cellPositions[i][j];
            if (
              x >= cell.x &&
              x <= cell.x + 162 &&
              y >= cell.y &&
              y <= cell.y + 114
            ) {
              this._cursorX = j;
              this._cursorY = i;
              this.updateHighlightPosition();
              if (!this.board[i][j] && this.currentPlayer === "X") {
                this.makeMove(j, i);
              }
              return;
            }
          }
        }
      }
    }

    initialize() {
      super.initialize();
      this.backgroundImageName = backgroundImage;
      this.createBackground();
      this.createBoard();
      this.currentPlayer = "X";
      this.isComputerThinking = false;
      this.boardSprites = [];
      this._cursorX = 0;
      this._cursorY = 0;
      this.highlightImageName = parameters["HighlightImage"];
      this.cellPositions = [
        [
          { x: 421, y: 117 },
          { x: 655, y: 121 },
          { x: 885, y: 129 },
        ],
        [
          { x: 372, y: 275 },
          { x: 612, y: 281 },
          { x: 846, y: 285 },
        ],
        [
          { x: 322, y: 438 },
          { x: 562, y: 441 },
          { x: 798, y: 450 },
        ],
      ];
      this.createHighlightSprite();
    }

    createHighlightSprite() {
      this.highlightSprite = new Sprite();
      this.highlightSprite.bitmap = ImageManager.loadPicture(
        this.highlightImageName,
      );
      this.addChild(this.highlightSprite);
      this.updateHighlightPosition();
    }

updateHighlightPosition() {
  if (
    this.cellPositions &&
    this._cursorY < this.cellPositions.length &&
    this._cursorX < this.cellPositions[this._cursorY].length
  ) {
    const pos = this.cellPositions[this._cursorY][this._cursorX];
    if (this.highlightSprite && this.highlightSprite.bitmap) {
      this.highlightSprite.x = pos.x;
      this.highlightSprite.y = pos.y;
    }
  }
}


    create() {
      super.create();
      this.createWindowLayer();
      this.createBoardWindows();
    }

    createBackground() {
      this.backgroundSprite = new Sprite();
      this.backgroundSprite.bitmap = ImageManager.loadPicture(
        this.backgroundImageName,
      );
      this.backgroundSprite.bitmap.addLoadListener(() => {
        this.backgroundSprite.x =
          (Graphics.width - this.backgroundSprite.width) / 2;
        this.backgroundSprite.y =
          (Graphics.height - this.backgroundSprite.height) / 2;
      });
      this.addChild(this.backgroundSprite);
    }

update() {
  super.update();

  if (this.gameOver) {
    return;
  }

  if (Input.isTriggered('cancel') || TouchInput.isCancelled()) {
    this.onCancel();
  }

      this.processCursorMovement();
      this.processMouseInput();
      this.updateBoardWindows();
      this.updateHighlightPosition();

      if (this.currentPlayer === "X" && !this.isComputerThinking) {
        this.processMouseInput();
        if (this.checkWin()) {
          this.endGame("win");
        } else if (this.checkTie()) {
          this.endGame("tie");
        }
      }

      if (this.currentPlayer === "O" && !this.isComputerThinking) {
        this.isComputerThinking = true;
        setTimeout(() => {
          this.computerMove();
          this.updateBoardWindows();
          this.isComputerThinking = false;
          if (this.checkWin()) {
            this.endGame("lose");
          } else if (this.checkTie()) {
            this.endGame("tie");
          } else {
            this.currentPlayer = "X";
          }
        }, 1000);
      }
    }

onCancel() {
  $gameSwitches.setValue(winSwitch, false);
  $gameSwitches.setValue(loseSwitch, false);
  $gameSwitches.setValue(tieSwitch, false);

  SceneManager.goto(Scene_Map);
}

    processCursorMovement() {
      if (Input.isRepeated("down")) {
        this.moveCursor(0, 1);
      } else if (Input.isRepeated("up")) {
        this.moveCursor(0, -1);
      } else if (Input.isRepeated("right")) {
        this.moveCursor(1, 0);
      } else if (Input.isRepeated("left")) {
        this.moveCursor(-1, 0);
      }

      if (Input.isTriggered("ok") && this.currentPlayer === "X") {
        this.selectCell();
      }
    }

    moveCursor(x, y) {
      this._cursorX = (this._cursorX + x + 3) % 3;
      this._cursorY = (this._cursorY + y + 3) % 3;
      this.updateHighlightPosition();
    }

    selectCell() {
      if (
        !this.board[this._cursorY][this._cursorX] &&
        this.currentPlayer === "X"
      ) {
        this.makeMove(this._cursorX, this._cursorY);
      }
    }

computerMove() {
  if (this.gameOver) {
    return;
  }
  let move = this.findWinningMove("O");
  if (!move) {
    move = this.findWinningMove("X");
    if (!move) {
      if (!this.board[1][1]) {
        move = { x: 1, y: 1 };
      } else {
        move = this.getRandomMove();
      }
    }
  }

  if (move) {
    this.board[move.y][move.x] = "O";
    if (oSound) {
        AudioManager.playSe({name: oSound, pan: 0, pitch: 100, volume: 90});
    }
  }
}


    findWinningMove(player) {
      for (let y = 0; y < 3; y++) {
        for (let x = 0; x < 3; x++) {
          if (!this.board[y][x]) {
            this.board[y][x] = player;
            if (this.checkWin()) {
              this.board[y][x] = null;
              return { x, y };
            }
            this.board[y][x] = null;
          }
        }
      }
      return null;
    }

    getRandomMove() {
      let availableMoves = [];
      for (let y = 0; y < 3; y++) {
        for (let x = 0; x < 3; x++) {
          if (!this.board[y][x]) {
            availableMoves.push({ x, y });
          }
        }
      }
      if (availableMoves.length > 0) {
        return availableMoves[
          Math.floor(Math.random() * availableMoves.length)
        ];
      }
      return null;
    }

    createBoard() {
      this.board = Array(3)
        .fill()
        .map(() => Array(3).fill(null));
    }

    createBoardWindows() {
      this.boardWindows = [];
      const cellWidth = 162;
      const cellHeight = 114;

      for (let i = 0; i < 3; i++) {
        for (let j = 0; j < 3; j++) {
          const pos = this.cellPositions[i][j];
          const window = new Window_Base(
            new Rectangle(pos.x, pos.y, cellWidth, cellHeight),
          );
          window.opacity = Number(parameters["BoardOpacity"]);
          this.addWindow(window);
          this.boardWindows.push(window);
        }
      }
    }

    clearBoardSprites() {
      if (this.boardSprites) {
        this.boardSprites.forEach((sprite) => sprite.destroy());
      }
      this.boardSprites = [];
    }

updateBoardWindows() {
    this.clearBoardSprites();

    for (let y = 0; y < 3; y++) {
        for (let x = 0; x < 3; x++) {
            const windowIndex = y * 3 + x;
            const window = this.boardWindows[windowIndex];

            if (this.board[y][x]) { 
                const sprite = new Sprite();
                const imageName = this.board[y][x] === "X" ? xImage : oImage;
                sprite.bitmap = ImageManager.loadPicture(imageName);
                sprite.bitmap.addLoadListener(() => {
                    if (sprite && sprite.bitmap) {
                        sprite.x = window.x + (window.width - sprite.bitmap.width) / 2;
                        sprite.y = window.y + (window.height - sprite.bitmap.height) / 2;
                        this.addChild(sprite);
                    }
                });

                this.boardSprites.push(sprite);
            }
        }
    }
}


    checkWin() {
      const lines = [
        [
          [0, 0],
          [0, 1],
          [0, 2],
        ],
        [
          [1, 0],
          [1, 1],
          [1, 2],
        ],
        [
          [2, 0],
          [2, 1],
          [2, 2],
        ],
        [
          [0, 0],
          [1, 0],
          [2, 0],
        ],
        [
          [0, 1],
          [1, 1],
          [2, 1],
        ],
        [
          [0, 2],
          [1, 2],
          [2, 2],
        ],
        [
          [0, 0],
          [1, 1],
          [2, 2],
        ],
        [
          [0, 2],
          [1, 1],
          [2, 0],
        ],
      ];

      for (const line of lines) {
        const [a, b, c] = line;
        if (
          this.board[a[0]][a[1]] &&
          this.board[a[0]][a[1]] === this.board[b[0]][b[1]] &&
          this.board[a[0]][a[1]] === this.board[c[0]][c[1]]
        ) {
          return true;
        }
      }
      return false;
    }

    checkTie() {
      return this.board.flat().every((cell) => cell != null);
    }

endGame(result) {
    this.gameOver = true;
    let message = "";

    switch (result) {
        case "win":
            message = this.customWinMessage || winMessage;
            $gameSwitches.setValue(winSwitch, true);
            $gameSwitches.setValue(loseSwitch, false);
            $gameSwitches.setValue(tieSwitch, false);
            break;
        case "lose":
            message = this.customLoseMessage || loseMessage;
            $gameSwitches.setValue(winSwitch, false);
            $gameSwitches.setValue(loseSwitch, true);
            $gameSwitches.setValue(tieSwitch, false);
            break;
        case "tie":
            message = this.customTieMessage || tieMessage;
            $gameSwitches.setValue(winSwitch, false);
            $gameSwitches.setValue(loseSwitch, false);
            $gameSwitches.setValue(tieSwitch, true);
            break;
    }

    if (this.showEndGameMessage) {
        $gameMessage.add(message);
    }


    setTimeout(() => {
        SceneManager.goto(Scene_Map);
    }, 500);
}



  }
})();
